from aiogram import BaseMiddleware
from aiogram.types import CallbackQuery, Message

from utils.lock import user_lock
from utils.throttle import debounce


class BusyLockMiddleware(BaseMiddleware):
    """Prevents concurrent actions per user and shows loading feedback."""

    async def __call__(self, handler, event, data):
        user = getattr(event, "from_user", None)
        if not user:
            return await handler(event, data)

        uid = user.id

        # Debounce callbacks/messages (anti double tap)
        if isinstance(event, CallbackQuery):
            if debounce.hit(f"cb:{uid}:{event.data}", window=0.6):
                # silently ignore quick repeats
                try:
                    await event.answer("⏳ دریافت شد...", show_alert=False)
                except Exception:
                    pass
                return

        # Busy lock
        if user_lock.locked(uid):
            # notify and block
            if isinstance(event, CallbackQuery):
                try:
                    await event.answer("⏳ لطفاً صبر کنید...", show_alert=False)
                except Exception:
                    pass
            elif isinstance(event, Message):
                try:
                    await event.answer("⏳ لطفاً صبر کنید...", disable_notification=True)
                except Exception:
                    pass
            return

        # acquire and run
        user_lock.acquire(uid)
        try:
            # show quick feedback for callbacks
            if isinstance(event, CallbackQuery):
                try:
                    await event.answer("⏳ در حال انجام...", show_alert=False)
                except Exception:
                    pass
            result = await handler(event, data)
            return result
        finally:
            user_lock.release(uid)



from aiogram import BaseMiddleware
from aiogram.types import Message, CallbackQuery
from repositories.users_repo import users_repo
from repositories.sellers_repo import sellers_repo
from config import settings

# router keys for isolation
ROUTER_ROLE_MAP = {
    "buyer": "buyer",
    "seller": "seller",
    "admin": "admin",
}

def _router_group_from_data(data: dict) -> str | None:
    # aiogram provides current_router in data in v3
    r = data.get("router")
    if not r:
        return None
    name = getattr(r, "name", "") or ""
    # names like 'handlers.buyer.menu' not set; so fallback: check parent package in handler module via handler object
    return name or None

class RoleGuardMiddleware(BaseMiddleware):
    """
    Ensures handlers run only in allowed user mode:
    - buyer handlers only in buyer mode
    - seller handlers only in seller mode
    - admin handlers only for admins
    Common handlers (/start, /buyer, /seller, /admin) are always allowed.
    """

    async def __call__(self, handler, event, data):
        user = getattr(event, "from_user", None)
        if not user:
            return await handler(event, data)

        uid = user.id

        # detect handler module
        h = data.get("handler")
        module = ""
        if h and hasattr(h, "__module__"):
            module = h.__module__
        elif hasattr(handler, "__module__"):
            module = handler.__module__

        # common is always allowed
        if ".handlers.common" in module:
            return await handler(event, data)

        # determine user mode
        u = await users_repo.get_user(uid)
        mode = (u.get("mode") if u else "buyer") or "buyer"

        # enforce admin only for admins
        if ".handlers.admin" in module:
            if uid not in settings.ADMIN_IDS:
                if isinstance(event, CallbackQuery):
                    try: await event.answer("⛔ دسترسی ندارید", show_alert=True)
                    except Exception: pass
                else:
                    try: await event.answer("⛔ دسترسی ندارید")
                    except Exception: pass
                return
            # also force mode admin
            return await handler(event, data)

        if ".handlers.seller" in module:
            # must be in seller mode AND seller active
            if mode != "seller":
                if isinstance(event, CallbackQuery):
                    try: await event.answer("🅱️ شما در حالت مشتری هستید. برای حالت فروشنده /seller بزنید.", show_alert=True)
                    except Exception: pass
                else:
                    await event.answer("🅱️ شما در حالت مشتری هستید. برای حالت فروشنده /seller بزنید.")
                return
            s = await sellers_repo.get_seller(uid)
            if not s or s.get("status") != "active":
                if isinstance(event, CallbackQuery):
                    try: await event.answer("⏳ شما فروشنده فعال نیستید. /seller_apply", show_alert=True)
                    except Exception: pass
                else:
                    await event.answer("⏳ شما فروشنده فعال نیستید. /seller_apply")
                return
            return await handler(event, data)

        if ".handlers.buyer" in module:
            if mode != "buyer":
                if isinstance(event, CallbackQuery):
                    try: await event.answer("🆂 شما در حالت فروشنده هستید. برای حالت مشتری /buyer بزنید.", show_alert=True)
                    except Exception: pass
                else:
                    await event.answer("🆂 شما در حالت فروشنده هستید. برای حالت مشتری /buyer بزنید.")
                return
            return await handler(event, data)

        return await handler(event, data)
