from db import fetchone, fetchall, execute


async def get_flower(flower_id: int):
    return await fetchone(
        "SELECT f.*, s.card_number as seller_card, s.card_owner as seller_owner "
        "FROM flowers f LEFT JOIN sellers s ON s.user_id=f.seller_id WHERE f.id=%s",
        (flower_id,)
    )


async def list_marketplace(page: int = 1, per_page: int = 6):
    offset = (page - 1) * per_page
    rows = await fetchall(
        "SELECT * FROM flowers WHERE is_active=1 AND stock>0 ORDER BY id DESC LIMIT %s OFFSET %s",
        (per_page, offset)
    )
    total = await fetchone("SELECT COUNT(*) as c FROM flowers WHERE is_active=1 AND stock>0")
    pages = max(1, (int(total['c'] or 0) + per_page - 1) // per_page)
    return rows, pages


async def dec_stock(flower_id: int, qty: int):
    await execute("UPDATE flowers SET stock=GREATEST(stock-%s,0) WHERE id=%s", (qty, flower_id))


async def inc_stock(flower_id: int, qty: int):
    await execute("UPDATE flowers SET stock=stock+%s WHERE id=%s", (qty, flower_id))


async def search_flowers_by_keyword(keyword: str, page: int = 1, per_page: int = 10):
    keyword = (keyword or "").strip().lower()
    if not keyword:
        return [], 1
    offset = (page - 1) * per_page
    rows = await fetchall(
        "SELECT id, seller_id, name, price_toman, stock, photo_file_id, is_active, normalized_name "
        "FROM flowers WHERE is_active=1 AND stock>0 AND normalized_name LIKE %s "
        "ORDER BY id DESC LIMIT %s OFFSET %s",
        (f"%{keyword}%", per_page, offset)
    )
    total_row = await fetchone(
        "SELECT COUNT(*) as c FROM flowers WHERE is_active=1 AND stock>0 AND normalized_name LIKE %s",
        (f"%{keyword}%",)
    )
    total = int(total_row["c"]) if total_row else 0
    pages = max(1, (total + per_page - 1) // per_page)
    return rows, pages


async def list_seller_flowers(seller_id:int, page:int=1, per_page:int=6):
    offset=(page-1)*per_page
    rows = await fetchall(
        "SELECT * FROM flowers WHERE seller_id=%s ORDER BY id DESC LIMIT %s OFFSET %s",
        (seller_id, per_page, offset)
    )
    total_row = await fetchone("SELECT COUNT(*) as c FROM flowers WHERE seller_id=%s", (seller_id,))
    total = int(total_row["c"]) if total_row else 0
    pages = max(1, (total + per_page - 1)//per_page)
    return rows, total, pages

async def set_flower_active(flower_id:int, is_active:int):
    return await execute("UPDATE flowers SET is_active=%s WHERE id=%s", (is_active, flower_id))

async def delete_flower(flower_id:int):
    return await execute("DELETE FROM flowers WHERE id=%s", (flower_id,))

async def update_flower_stock(flower_id:int, stock:int):
    return await execute("UPDATE flowers SET stock=%s WHERE id=%s", (stock, flower_id))

async def update_flower_price(flower_id:int, price:int):
    return await execute("UPDATE flowers SET price_toman=%s WHERE id=%s", (price, flower_id))

