from aiogram import Router, F
from aiogram.types import Message, CallbackQuery

from repositories.orders_repo import list_orders_by_seller, get_order, set_status, mark_shipped, mark_delivered
from keyboards.seller import seller_orders_menu_kb, seller_order_actions_kb
from config import settings
from utils.filters import ModeFilter

router = Router()
router.message.filter(ModeFilter('seller'))
router.callback_query.filter(ModeFilter('seller'))


@router.message(F.text == "📦 سفارش‌ها")
async def seller_orders_entry(message: Message):
    await message.answer("📦 مدیریت سفارش‌ها:", reply_markup=seller_orders_menu_kb())


@router.callback_query(F.data.startswith("sord:"))
async def seller_orders_list(call: CallbackQuery):
    mode = call.data.split(":")[1]
    uid = call.from_user.id
    status = None
    title = "همه سفارش‌ها"
    if mode == "new":
        status = "pending_seller_approval"
        title = "سفارش‌های جدید (رسید ارسال شده)"
    elif mode == "ship":
        status = "ready_to_ship"
        title = "آماده ارسال"

    rows = await list_orders_by_seller(uid, status=status, limit=20)
    if not rows:
        return await call.message.answer(f"✅ {title} خالی است.")
    text = f"📦 <b>{title}</b>\n\n"
    for r in rows:
        text += f"#{r['id']} | {r['status']} | {int(r['total_price'] or 0):,} تومان\n"
    text += "\nبرای مدیریت هر سفارش: /order <id>"
    await call.message.answer(text)
    await call.answer()


@router.message(F.text.regexp(r"^/order\s+\d+$"))
async def seller_order_view(message: Message):
    oid = int(message.text.split()[1])
    order = await get_order(oid)
    if not order or order["seller_id"] != message.from_user.id:
        return await message.answer("❌ سفارش یافت نشد.")
    txt = (
        f"📦 <b>سفارش #{oid}</b>\n"
        f"وضعیت: <b>{order['status']}</b>\n"
        f"تعداد: <b>{order['qty']}</b>\n"
        f"مبلغ: <b>{int(order['total_price'] or 0):,}</b> تومان\n"
    )
    await message.answer(txt, reply_markup=seller_order_actions_kb(oid, order["status"]))


@router.callback_query(F.data.startswith("rec_ok:"))
async def receipt_ok(call: CallbackQuery):
    oid = int(call.data.split(":")[1])
    order = await get_order(oid)
    if not order or order["seller_id"] != call.from_user.id:
        return await call.answer("دسترسی ندارید", show_alert=True)
    await set_status(oid, "accepted")
    from keyboards.orders import request_location_kb
    await call.bot.send_message(order["buyer_id"], f"✅ رسید سفارش #{oid} تایید شد. لطفاً لوکیشن ارسال کنید.", reply_markup=request_location_kb())
    await call.message.answer("✅ رسید تایید شد.")
    await call.answer()


@router.callback_query(F.data.startswith("rec_no:"))
async def receipt_no(call: CallbackQuery):
    oid = int(call.data.split(":")[1])
    order = await get_order(oid)
    if not order or order["seller_id"] != call.from_user.id:
        return await call.answer("دسترسی ندارید", show_alert=True)
    await set_status(oid, "rejected")
    await call.bot.send_message(order["buyer_id"], f"❌ رسید سفارش #{oid} رد شد.")
    await call.message.answer("❌ رسید رد شد.")
    await call.answer()


@router.callback_query(F.data.startswith("ship:"))
async def mark_ship(call: CallbackQuery):
    oid = int(call.data.split(":")[1])
    order = await get_order(oid)
    if not order or order["seller_id"] != call.from_user.id:
        return await call.answer("دسترسی ندارید", show_alert=True)
    await mark_shipped(oid)
    await call.bot.send_message(order["buyer_id"], f"🚚 سفارش #{oid} ارسال شد و در مسیر است.")
    await call.message.answer("✅ وضعیت سفارش shipped شد.")
    await call.answer()


@router.callback_query(F.data.startswith("deliv:"))
async def mark_deliv(call: CallbackQuery):
    oid = int(call.data.split(":")[1])
    order = await get_order(oid)
    if not order or order["seller_id"] != call.from_user.id:
        return await call.answer("دسترسی ندارید", show_alert=True)
    await mark_delivered(oid)
    from keyboards.orders import buyer_delivery_confirm_kb
    await call.bot.send_message(order["buyer_id"], f"📦 سفارش #{oid} تحویل شد. لطفاً تایید کنید.", reply_markup=buyer_delivery_confirm_kb(oid))
    await call.message.answer("✅ وضعیت سفارش delivered شد.")
    await call.answer()
