
from aiogram import Router, F
from aiogram.types import Message, CallbackQuery
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import StatesGroup, State

from services.role_service import role_service
from services.need_service import need_service
from keyboards.seller import seller_needs_list_kb, seller_need_card_kb
from utils.filters import ModeFilter

router = Router()
router.message.filter(ModeFilter('seller'))
router.callback_query.filter(ModeFilter('seller'))


class OfferQuick(StatesGroup):
    need_id = State()
    photo = State()
    price = State()
    caption = State()


@router.message(F.text == "📌 نیازمندی‌ها")
async def needs_home(message: Message):
    if not await role_service.can_be_seller(message.from_user.id):
        return await message.answer("⏳ شما هنوز فروشنده فعال نیستید. منتظر تایید ادمین بمانید.")
    await show_needs_page(message, page=1)


async def show_needs_page(message_or_call, page:int=1):
    per_page = 6
    data = await need_service.get_open_needs(page=page, per_page=per_page)
    rows = data["rows"]
    pages = data["pages"]
    if not rows:
        text = "✅ نیازمندی باز نداریم."
        kb = seller_needs_list_kb(page, pages)
    else:
        text = "📌 <b>نیازمندی‌های باز</b>\n\n"
        for r in rows:
            short = (r["text"][:60] + "…") if len(r["text"]) > 60 else r["text"]
            budget = f" | 💰 {int(r['budget']):,}" if r.get("budget") else ""
            city = f" | 🏙 {r['city']}" if r.get("city") else ""
            text += f"#{r['id']} {short}{city}{budget}\n"
        text += f"\nصفحه {page}/{pages}"
        kb = seller_needs_list_kb(page, pages)
    if isinstance(message_or_call, CallbackQuery):
        await message_or_call.message.edit_text(text, reply_markup=kb)
    else:
        await message_or_call.answer(text, reply_markup=kb)


@router.callback_query(F.data.startswith("sneeds:"))
async def needs_nav(call: CallbackQuery):
    page = int(call.data.split(":")[1])
    await show_needs_page(call, page=page)
    await call.answer()


@router.callback_query(F.data.startswith("need:view:"))
async def need_view(call: CallbackQuery):
    need_id = int(call.data.split(":")[2])
    need = await need_service.get_need(need_id)
    if not need or need["status"] != "open":
        return await call.answer("این نیازمندی بسته شده است.", show_alert=True)

    txt = f"📌 <b>نیازمندی #{need_id}</b>\n\n{need['text']}"
    if need.get("city"):
        txt += f"\n🏙 شهر: <b>{need['city']}</b>"
    if need.get("budget"):
        txt += f"\n💰 بودجه: <b>{int(need['budget']):,}</b> تومان"
    await call.message.answer(txt, reply_markup=seller_need_card_kb(need_id))
    await call.answer()


@router.callback_query(F.data.startswith("need:offer_quick:"))
async def offer_quick_start(call: CallbackQuery, state: FSMContext):
    need_id = int(call.data.split(":")[2])
    need = await need_service.get_need(need_id)
    if not need or need["status"] != "open":
        return await call.answer("این نیازمندی بسته شده است.", show_alert=True)

    await state.clear()
    await state.update_data(need_id=need_id)
    await state.set_state(OfferQuick.photo)
    await call.message.answer("📷 عکس محصول پیشنهادی را ارسال کنید:")
    await call.answer()


@router.message(OfferQuick.photo, F.photo)
async def offer_photo(message: Message, state: FSMContext):
    await state.update_data(photo_file_id=message.photo[-1].file_id)
    await state.set_state(OfferQuick.price)
    await message.answer("💰 قیمت (تومان) را فقط عدد ارسال کنید (مثلاً 2500000):")


@router.message(OfferQuick.price)
async def offer_price(message: Message, state: FSMContext):
    txt = (message.text or "").strip().replace(",", "")
    if not txt.isdigit():
        return await message.answer("⚠️ فقط عدد ارسال کنید.")
    await state.update_data(price=int(txt))
    await state.set_state(OfferQuick.caption)
    await message.answer("📝 توضیح کوتاه (اختیاری). اگر ندارید /skip بزنید.")


@router.message(OfferQuick.caption, F.text == "/skip")
async def offer_skip(message: Message, state: FSMContext):
    await finalize_offer(message, state, caption=None)


@router.message(OfferQuick.caption)
async def offer_caption(message: Message, state: FSMContext):
    await finalize_offer(message, state, caption=(message.text or "").strip())


async def finalize_offer(message: Message, state: FSMContext, caption: str | None):
    data = await state.get_data()
    need_id = data["need_id"]
    offer_id = await need_service.create_offer(
        need_id=need_id,
        seller_id=message.from_user.id,
        photo_file_id=data["photo_file_id"],
        price=data["price"],
        caption=caption
    )
    await message.answer("✅ پیشنهاد شما ثبت و برای مشتری ارسال شد.", reply_markup=None)
    await state.clear()


@router.callback_query(F.data == "need:pick")
async def need_pick(call: CallbackQuery, state: FSMContext):
    await state.set_state(OfferQuick.need_id)
    await call.message.answer("شماره نیازمندی را ارسال کنید (مثلاً 12):")
    await call.answer()

@router.message(OfferQuick.need_id)
async def need_pick_msg(message: Message, state: FSMContext):
    if not (message.text or "").strip().isdigit():
        return await message.answer("⚠️ فقط عدد ارسال کنید.")
    need_id = int(message.text.strip())
    need = await need_service.get_need(need_id)
    await state.clear()
    if not need or need["status"] != "open":
        return await message.answer("❌ نیازمندی یافت نشد یا بسته شده است.")
    txt = f"📌 <b>نیازمندی #{need_id}</b>\n\n{need['text']}"
    if need.get("city"):
        txt += f"\n🏙 شهر: <b>{need['city']}</b>"
    if need.get("budget"):
        txt += f"\n💰 بودجه: <b>{int(need['budget']):,}</b> تومان"
    await message.answer(txt, reply_markup=seller_need_card_kb(need_id))
