
from aiogram import Router, F
from aiogram.types import Message, CallbackQuery
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import StatesGroup, State

from services.role_service import role_service
from services.listing_service import listing_service
from keyboards.seller import seller_listings_home_kb, seller_listing_card_kb
from utils.filters import ModeFilter

router = Router()
router.message.filter(ModeFilter('seller'))
router.callback_query.filter(ModeFilter('seller'))


class ListingEdit(StatesGroup):
    flower_id = State()
    stock = State()
    price = State()


@router.message(F.text == "📋 آگهی‌های من")
async def listings_home(message: Message):
    if not await role_service.can_be_seller(message.from_user.id):
        return await message.answer("⏳ شما هنوز فروشنده فعال نیستید. منتظر تایید ادمین بمانید.")
    await show_listings(message, page=1)


async def show_listings(message_or_call, page:int=1):
    per_page = 6
    uid = message_or_call.from_user.id if hasattr(message_or_call, "from_user") else message_or_call.message.from_user.id
    data = await listing_service.get_seller_listings(uid, page=page, per_page=per_page)
    rows = data["rows"]
    pages = data["pages"]

    if not rows:
        text = "❌ هنوز هیچ آگهی ثبت نکرده‌اید.\nبرای ثبت روی «➕ افزودن گل» بزنید."
        kb = seller_listings_home_kb(page, pages)
        if isinstance(message_or_call, CallbackQuery):
            await message_or_call.message.edit_text(text, reply_markup=kb)
        else:
            await message_or_call.answer(text, reply_markup=kb)
        return

    # show summary + then cards
    text = f"🌸 <b>آگهی‌های شما</b> (صفحه {page}/{pages})"
    kb = seller_listings_home_kb(page, pages)
    if isinstance(message_or_call, CallbackQuery):
        await message_or_call.message.edit_text(text, reply_markup=kb)
    else:
        await message_or_call.answer(text, reply_markup=kb)

    for f in rows:
        cap = listing_service.flower_caption(f, compact=True)
        await (message_or_call.bot if hasattr(message_or_call,"bot") else message_or_call.message.bot).send_photo(
            uid,
            f["photo_file_id"],
            caption=cap,
            reply_markup=seller_listing_card_kb(f["id"], f["is_active"])
        )


@router.callback_query(F.data.startswith("slist:"))
async def listings_nav(call: CallbackQuery):
    page = int(call.data.split(":")[1])
    await show_listings(call, page=page)
    await call.answer()


@router.callback_query(F.data.startswith("lst:toggle:"))
async def listing_toggle(call: CallbackQuery):
    fid = int(call.data.split(":")[2])
    f = await listing_service.get_flower(fid)
    if not f or f["seller_id"] != call.from_user.id:
        return await call.answer("دسترسی ندارید.", show_alert=True)
    new_active = 0 if int(f["is_active"]) == 1 else 1
    await listing_service.set_active(fid, new_active)
    await call.answer("✅ انجام شد")
    await call.message.answer("✅ وضعیت آگهی تغییر کرد. (برای دیدن لیست مجدد «🌸 آگهی‌های من» را بزنید.)")


@router.callback_query(F.data.startswith("lst:del:"))
async def listing_delete(call: CallbackQuery):
    fid = int(call.data.split(":")[2])
    f = await listing_service.get_flower(fid)
    if not f or f["seller_id"] != call.from_user.id:
        return await call.answer("دسترسی ندارید.", show_alert=True)
    await listing_service.delete_flower(fid)
    await call.answer("✅ حذف شد")
    await call.message.answer("🗑 آگهی حذف شد.")


@router.callback_query(F.data.startswith("lst:stock:"))
async def listing_stock_start(call: CallbackQuery, state: FSMContext):
    fid = int(call.data.split(":")[2])
    f = await listing_service.get_flower(fid)
    if not f or f["seller_id"] != call.from_user.id:
        return await call.answer("دسترسی ندارید.", show_alert=True)
    await state.update_data(flower_id=fid)
    await state.set_state(ListingEdit.stock)
    await call.message.answer("📦 موجودی جدید را فقط عدد ارسال کنید:")
    await call.answer()


@router.message(ListingEdit.stock)
async def listing_stock_save(message: Message, state: FSMContext):
    d = await state.get_data()
    fid = int(d["flower_id"])
    txt = (message.text or "").strip().replace(",", "")
    if not txt.isdigit():
        return await message.answer("⚠️ فقط عدد.")
    await listing_service.update_stock(fid, int(txt))
    await message.answer("✅ موجودی آپدیت شد.")
    await state.clear()


@router.callback_query(F.data.startswith("lst:price:"))
async def listing_price_start(call: CallbackQuery, state: FSMContext):
    fid = int(call.data.split(":")[2])
    f = await listing_service.get_flower(fid)
    if not f or f["seller_id"] != call.from_user.id:
        return await call.answer("دسترسی ندارید.", show_alert=True)
    await state.update_data(flower_id=fid)
    await state.set_state(ListingEdit.price)
    await call.message.answer("💰 قیمت جدید (تومان) را فقط عدد ارسال کنید:")
    await call.answer()


@router.message(ListingEdit.price)
async def listing_price_save(message: Message, state: FSMContext):
    d = await state.get_data()
    fid = int(d["flower_id"])
    txt = (message.text or "").strip().replace(",", "")
    if not txt.isdigit():
        return await message.answer("⚠️ فقط عدد.")
    await listing_service.update_price(fid, int(txt))
    await message.answer("✅ قیمت آپدیت شد.")
    await state.clear()