from aiogram import Router, F
from aiogram.filters import CommandStart, Command
from aiogram.types import Message, ReplyKeyboardMarkup, KeyboardButton, ReplyKeyboardRemove
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import StatesGroup, State

from services.role_service import role_service
from repositories.users_repo import users_repo
from repositories.sellers_repo import sellers_repo
from keyboards.buyer import buyer_home_kb
from keyboards.seller import seller_home_kb
from keyboards.admin import admin_home_kb
from config import settings

router = Router()


def role_badge(mode: str) -> str:
    # A=admin, B=buyer, S=seller
    return {"admin": "🅰️", "buyer": "🅱️", "seller": "🆂"}.get(mode, "🅱️")


def phone_request_kb():
    return ReplyKeyboardMarkup(
        keyboard=[[KeyboardButton(text="📱 ارسال شماره تلفن", request_contact=True)]],
        resize_keyboard=True,
        one_time_keyboard=True
    )


async def show_home(message: Message):
    mode = await role_service.get_user_mode(message.from_user.id)
    badge = role_badge(mode)

    if mode == "admin":
        return await message.answer(
            f"{badge} <b>پنل ادمین</b>",
            reply_markup=admin_home_kb()
        )
    if mode == "seller":
        return await message.answer(
            f"{badge} <b>پنل فروشنده</b>",
            reply_markup=seller_home_kb()
        )
    # buyer default
    return await message.answer(
        f"{badge} <b>پنل مشتری</b>",
        reply_markup=buyer_home_kb()
    )


@router.message(CommandStart())
async def start(message: Message):
    uid = message.from_user.id
    await users_repo.ensure_user(uid)

    phone = await users_repo.get_phone(uid)
    if not phone:
        await message.answer(
            "برای شروع، لطفاً شماره موبایل خود را ارسال کنید 👇",
            reply_markup=phone_request_kb()
        )
        return

    # ✅ admin always lands on admin panel on /start
    if uid in settings.ADMIN_IDS:
        await users_repo.set_mode(uid, "admin")
        await show_home(message)
        return

    await show_home(message)
@router.message(F.contact)
async def on_contact(message: Message):
    # Telegram guarantees contact.user_id equals sender when request_contact=True
    uid = message.from_user.id
    if not message.contact or message.contact.user_id != uid:
        # ignore чужой contact
        return await message.answer("⚠️ لطفاً فقط شماره خودتان را ارسال کنید.", reply_markup=phone_request_kb())

    phone = message.contact.phone_number
    await users_repo.set_phone(uid, phone)

    # if approved seller exists, store phone also in sellers table
    try:
        await sellers_repo.set_phone(uid, phone)
    except Exception:
        pass

    # remove contact keyboard and show home immediately (no extra dialogs)
    await message.answer("✅ ثبت شد.", reply_markup=ReplyKeyboardRemove())
    await show_home(message)


@router.message(Command("buyer"))
async def cmd_buyer(message: Message):
    await users_repo.set_mode(message.from_user.id, "buyer")
    await show_home(message)


@router.message(Command("seller"))
async def cmd_seller(message: Message, state: FSMContext):
    # start seller registration wizard or show status
    s = await sellers_repo.get_seller(message.from_user.id)
    if s:
        if s.get("status") == "active":
            await role_service.set_mode(message.from_user.id, "seller")
            return await message.answer("✅ شما فروشنده تایید شده هستید. وارد پنل فروشنده شدید.", reply_markup=seller_home_kb())
        if s.get("status") == "pending":
            return await message.answer("⏳ درخواست فروشندگی شما ثبت شده و در انتظار تایید مدیر است.", reply_markup=seller_pending_kb())
        if s.get("status") == "rejected":
            return await message.answer("❌ درخواست فروشندگی شما رد شده است. برای درخواست مجدد /seller را بزنید.", reply_markup=buyer_home_kb())

    # begin registration
    return await message.answer("🧑‍🌾 برای درخواست فروشندگی، دستور زیر را بزنید:\n/seller_apply", reply_markup=buyer_home_kb())

@router.message(Command("adminpanel"))
async def cmd_adminpanel(message: Message):
    if message.from_user.id not in settings.ADMIN_IDS:
        return await message.answer("⛔️ دسترسی ندارید.")
    await users_repo.set_mode(message.from_user.id, "admin")
    await show_home(message)


@router.message(Command("role"))
async def role_help(message: Message):
    role = await role_service.get_effective_role(message.from_user.id)
    cur = await role_service.get_user_mode(message.from_user.id)
    await message.answer(
        f"🔰 نقش شما: <b>{role}</b>\n"
        f"🔁 حالت فعلی: <b>{cur}</b> {role_badge(cur)}\n\n"
        "📌 تغییر حالت فقط با دستور:\n"
        "• /buyer  → حالت مشتری\n"
        "• /seller → حالت فروشنده (اگر تایید شده)\n"
        "• /adminpanel → پنل ادمین (فقط ادمین)\n"
    )

# ---------- Seller Apply (Simple) ----------
class SellerApply(StatesGroup):
    shop = State()
    city = State()

@router.message(Command("seller_apply"))
async def seller_apply_start(message: Message, state: FSMContext):
    s = await sellers_repo.get_seller(message.from_user.id)
    if s and s.get("status") == "active":
        await role_service.set_mode(message.from_user.id, "seller")
        return await message.answer("✅ شما فروشنده تایید شده هستید. وارد پنل فروشنده شدید.", reply_markup=seller_home_kb())
    if s and s.get("status") == "pending":
        return await message.answer("⏳ درخواست شما قبلاً ثبت شده و در انتظار تایید مدیر است.", reply_markup=seller_pending_kb())
    await state.clear()
    await state.set_state(SellerApply.shop)
    await message.answer("🏪 نام فروشگاه/گلخانه را ارسال کنید:")

@router.message(SellerApply.shop)
async def seller_apply_shop(message: Message, state: FSMContext):
    shop = (message.text or "").strip()
    if len(shop) < 2:
        return await message.answer("⚠️ نام کوتاه است. دوباره ارسال کنید:")
    await state.update_data(shop=shop)
    await state.set_state(SellerApply.city)
    await message.answer("🏙 شهر را ارسال کنید:")

@router.message(SellerApply.city)
async def seller_apply_city(message: Message, state: FSMContext):
    city = (message.text or "").strip()
    if len(city) < 2:
        return await message.answer("⚠️ شهر کوتاه است. دوباره ارسال کنید:")
    data = await state.get_data()
    shop = data.get("shop")
    u = await users_repo.get_user(message.from_user.id)
    phone = u.get('phone') if u else None

    await sellers_repo.create_or_update(message.from_user.id, shop=shop, city=city, status="pending", phone=phone)
    await state.clear()
    await message.answer("✅ درخواست فروشندگی ثبت شد و در انتظار تایید مدیر است.", reply_markup=seller_pending_kb())
    # notify admins
    for admin in settings.ADMIN_IDS:
        try:
            await message.bot.send_message(admin, f"🧑‍🌾 درخواست فروشندگی جدید\nUser: {message.from_user.id}\nShop: {shop}\nCity: {city}")
        except Exception:
            pass