
from aiogram import Router, F
from aiogram.types import Message, CallbackQuery
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import StatesGroup, State

from repositories.orders_repo import get_active_order, list_orders, set_status
from services.order_service import next_step, order_caption
from keyboards.orders import active_order_kb, orders_menu_kb
from keyboards.buyer import buyer_main_menu

from utils.filters import ModeFilter

router = Router()
router.message.filter(ModeFilter('buyer'))
router.callback_query.filter(ModeFilter('buyer'))

class BuyFlow(StatesGroup):
    receipt = State()
    address = State()

@router.message(F.text == "📦 سفارش‌های من")
async def orders_entry(message: Message):
    await message.answer("📦 مدیریت سفارش‌ها:", reply_markup=orders_menu_kb())

@router.callback_query(F.data.startswith("ord_menu:"))
async def orders_menu_pick(call: CallbackQuery, state:FSMContext):
    mode = call.data.split(":")[1]
    uid = call.from_user.id

    if mode == "active":
        order = await get_active_order(uid)
        if not order:
            await call.message.answer("✅ سفارش فعالی ندارید.", reply_markup=buyer_main_menu())
            return await call.answer()
        step = next_step(order)
        txt = order_caption(order)
        if step == "receipt": txt += "\n📷 مرحله بعد: ارسال عکس رسید"
        elif step == "location": txt += "\n📍 مرحله بعد: ارسال لوکیشن"
        elif step == "address": txt += "\n🏠 مرحله بعد: ارسال آدرس"
        can_cancel = order["status"] == "pending_payment" and not order.get("receipt_file_id")
        await call.message.answer(txt, reply_markup=active_order_kb(order["id"], can_cancel=can_cancel, can_resume=True))
        return await call.answer()

    status_map = {"completed":"completed", "cancelled":"cancelled"}
    if mode in status_map:
        rows = await list_orders(uid, status=status_map[mode], limit=30)
    else:
        rows = await list_orders(uid, status=None, limit=30)
    if not rows:
        await call.message.answer("❌ موردی یافت نشد.", reply_markup=buyer_main_menu())
        return await call.answer()
    text="📋 <b>لیست سفارش‌ها</b>\n\n"
    for r in rows:
        text += f"#{r['id']} | {r.get('status')} | {int(r.get('total_price') or 0):,} تومان\n"
    await call.message.answer(text, reply_markup=buyer_main_menu())
    await call.answer()

@router.callback_query(F.data.startswith("ord_refresh:"))
async def active_refresh(call: CallbackQuery):
    oid = int(call.data.split(":")[1])
    order = await get_active_order(call.from_user.id)
    if not order or int(order["id"]) != oid:
        return await call.answer("سفارش فعال یافت نشد", show_alert=True)
    step = next_step(order)
    txt = order_caption(order)
    if step == "receipt": txt += "\n📷 مرحله بعد: ارسال عکس رسید"
    elif step == "location": txt += "\n📍 مرحله بعد: ارسال لوکیشن"
    elif step == "address": txt += "\n🏠 مرحله بعد: ارسال آدرس"
    can_cancel = order["status"] == "pending_payment" and not order.get("receipt_file_id")
    await call.message.edit_text(txt, reply_markup=active_order_kb(order["id"], can_cancel=can_cancel, can_resume=True))
    await call.answer()

@router.callback_query(F.data.startswith("ord_cancel:"))
async def active_cancel(call: CallbackQuery):
    oid = int(call.data.split(":")[1])
    order = await get_active_order(call.from_user.id)
    if not order or int(order["id"]) != oid:
        return await call.answer("دسترسی ندارید", show_alert=True)
    if order["status"] != "pending_payment" or order.get("receipt_file_id"):
        return await call.answer("❌ امکان لغو نیست.", show_alert=True)
    await set_status(oid, "cancelled")
    await call.message.edit_text("✅ سفارش لغو شد.")
    await call.answer()

@router.callback_query(F.data.startswith("ord_resume:"))
async def active_resume(call: CallbackQuery, state: FSMContext):
    oid = int(call.data.split(":")[1])
    order = await get_active_order(call.from_user.id)
    if not order or int(order["id"]) != oid:
        return await call.answer("دسترسی ندارید", show_alert=True)
    step = next_step(order)
    await state.update_data(order_id=oid)
    if step == "receipt":
        await state.set_state(BuyFlow.receipt)
        await call.message.answer("📷 لطفاً عکس رسید را ارسال کنید:")
    elif step == "location":
        await call.message.answer("📍 لطفاً لوکیشن را ارسال کنید (با دکمه ارسال لوکیشن در تلگرام):")
    elif step == "address":
        await state.set_state(BuyFlow.address)
        await call.message.answer("🏠 لطفاً آدرس را ارسال کنید:")
    else:
        await call.message.answer("✅ سفارش شما در حال پردازش است.")
    await call.answer()
