
from aiogram import Router, F
from aiogram.types import Message, CallbackQuery
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import StatesGroup, State

from services.listing_service import get_marketplace_page, get_search_page, flower_caption
from utils.digits import format_money
from keyboards.common import marketplace_nav_kb, flower_card_kb, marketplace_list_kb, marketplace_card_kb, search_mode_kb_v7
from keyboards.buyer import buyer_buy_kb

from utils.filters import ModeFilter

router = Router()
router.message.filter(ModeFilter('buyer'))
router.callback_query.filter(ModeFilter('buyer'))

PER_PAGE = 6

class SearchFlow(StatesGroup):
    query = State()

@router.message(F.text == "🏪 همه آگهی‌های فروش")
async def mp_all(message: Message, state: FSMContext):
    await state.clear()
    await message.answer("⏳ در حال دریافت آگهی‌ها ...")
    await _send_marketplace(message, page=1, mode="newest")

@router.message(F.text == "🔎 جستجوی پیشرفته")
async def mp_search_entry(message: Message, state: FSMContext):
    await state.clear()
    await message.answer("🔎 روش جستجو را انتخاب کنید:", reply_markup=search_mode_kb_v7())

@router.callback_query(F.data.startswith("mpmode:"))
async def mp_mode_pick(call: CallbackQuery, state: FSMContext):
    mode = call.data.split(":")[1]
    if mode == "newest":
        await call.message.answer("⏳ در حال دریافت آگهی‌ها ...")
        await _send_marketplace(call, page=1, mode="newest")
    elif mode == "name":
        await state.set_state(SearchFlow.query)
        await call.message.answer("✍️ نام گل را وارد کنید (مثلاً رز):")
    await call.answer()

@router.message(SearchFlow.query)
async def mp_search_name(message: Message, state: FSMContext):
    q = (message.text or "").strip()
    if len(q) < 2:
        return await message.answer("⚠️ حداقل 2 حرف وارد کنید.")
    await message.answer("⏳ در حال جستجو ...")
    await _send_marketplace(message, page=1, mode="name", q=q)
    await state.clear()

@router.callback_query(F.data.startswith("mp:"))
async def mp_nav(call: CallbackQuery):
    _, mode, page, q = call.data.split(":", 3)
    page = int(page)
    await _send_marketplace(call, page=page, mode=mode, q=q)
    await call.answer()

@router.callback_query(F.data.startswith("viewi:"))
async def mp_view_card_from_list(call: CallbackQuery):
    # view card by index from the compact list page
    _, idx, mode, page, q = call.data.split(":", 4)
    idx = int(idx); page = int(page)
    rows, pages = await list_flowers(mode=mode, q=q or "", page=page, per_page=PER_PAGE)
    if not rows or idx < 0 or idx >= len(rows):
        return await call.answer("یافت نشد", show_alert=True)
    f = rows[idx]

    name = (f.get("name") or "").strip()
    key = ((f.get("normalized_name") or name).split() or [name])[0].strip().lower()
    if not key:
        key = name[:10].strip().lower()

    cap = flower_caption(f)
    kb = marketplace_card_kb(int(f["id"]), int(f.get("seller_id") or 0), page, pages, mode=mode, q=q or "", name_key=key, idx=idx)
    await call.message.answer_photo(f["photo_file_id"], caption=cap, reply_markup=kb)
    await call.answer()

@router.callback_query(F.data.startswith("cardnav:"))
async def mp_card_nav(call: CallbackQuery):
    # navigate within cards on the same list page (prev/next item)
    _, idx, mode, page, q = call.data.split(":", 4)
    idx = int(idx); page = int(page)
    rows, pages = await list_flowers(mode=mode, q=q or "", page=page, per_page=PER_PAGE)
    if not rows:
        return await call.answer("لیست خالی است", show_alert=True)
    if idx < 0:
        idx = 0
    if idx >= len(rows):
        idx = len(rows) - 1
    f = rows[idx]

    name = (f.get("name") or "").strip()
    key = ((f.get("normalized_name") or name).split() or [name])[0].strip().lower()
    if not key:
        key = name[:10].strip().lower()

    cap = flower_caption(f)
    kb = marketplace_card_kb(int(f["id"]), int(f.get("seller_id") or 0), page, pages, mode=mode, q=q or "", name_key=key, idx=idx)
    await call.message.edit_media(
        media=InputMediaPhoto(media=f["photo_file_id"], caption=cap),
        reply_markup=kb
    )
    await call.answer()


async def mp_similar_list(call: CallbackQuery):
    # show similar products list by keyword
    _, key, mode, page, q = call.data.split(":", 4)
    key = key.strip().lower()
    from services.listing_service import get_similar_page
    rows, total, pages = await get_similar_page(key, page=1, per_page=PER_PAGE)
    if not rows:
        return await call.answer("مشابهی پیدا نشد", show_alert=True)

    lines=[]
    for i,f in enumerate(rows, start=1):
        price = format_money(int(f.get("price_toman") or 0))
        name = f.get("name") or "-"
        stock = int(f.get("stock") or 0)
        lines.append(f"{i}) 🌸 <b>{name}</b> | 💰 <b>{price}</b> | 📦 {stock}")
    text = f"🔎 <b>محصولات مشابه: {key}</b>\n\n" + "\n".join(lines) + f"\n\n📄 1/{pages}"
    kb = marketplace_list_kb(rows, 1, pages, mode="name", q=key)
    await call.message.answer(text, reply_markup=kb)
    await call.answer()

@router.callback_query(F.data.startswith("viewnav:"))
async def mp_viewnav(call: CallbackQuery):
    # view next/prev within the same list page is not implemented (heavy). kept for future.
    await call.answer("از لیست صفحه‌بندی استفاده کنید ✅", show_alert=True)

async def _send_marketplace(message_or_call, page:int=1, mode:str="newest", q:str=""):
    if mode == "name":
        rows, total, pages = await get_search_page(q, page=page, per_page=PER_PAGE)
    else:
        rows, total, pages = await get_marketplace_page(page=page, per_page=PER_PAGE)

    if not rows:
        if hasattr(message_or_call, "message"):
            return await message_or_call.message.answer("❌ آگهی‌ای یافت نشد.", reply_markup=buyer_buy_kb())
        return await message_or_call.answer("❌ آگهی‌ای یافت نشد.", show_alert=True)

    uid = message_or_call.from_user.id if hasattr(message_or_call, "from_user") else message_or_call.message.from_user.id
    bot = message_or_call.bot if hasattr(message_or_call, "bot") else message_or_call.message.bot

    # Compact list (no photos) for best mobile UX
    lines=[]
    for i,f in enumerate(rows, start=1):
        price = format_money(int(f.get("price_toman") or 0))
        name = f.get("name") or "-"
        stock = int(f.get("stock") or 0)
        lines.append(f"{i}) 🌸 <b>{name}</b> | 💰 <b>{price}</b> | 📦 {stock}")
    text = "🛒 <b>آگهی‌های فروش</b>\n\n" + "\n".join(lines) + f"\n\n📄 صفحه {page}/{pages}"
    kb = marketplace_list_kb(rows, page, pages, mode=mode, q=q or "")
    await bot.send_message(uid, text, reply_markup=kb)


# ---------- Buy Flow (v7.4) ----------
from aiogram.types import ReplyKeyboardMarkup, KeyboardButton
from repositories.flow_repo import flow_lock
from repositories.flow_repo import is_busy, set_busy, clear_busy
from repositories.flowers_repo import get_flower, dec_stock, inc_stock
from repositories.orders_repo import create_order, set_receipt, get_order, set_location, set_address
from keyboards.orders import receipt_actions_kb, request_location_kb, buyer_delivery_confirm_kb
from keyboards.common import choose_qty_kb
from utils.format import money

class BuyFlow(StatesGroup):
    qty = State()
    receipt = State()
    address = State()

@router.callback_query(F.data.startswith("buy:"))
async def buy_start(call: CallbackQuery, state: FSMContext):
    if await is_busy(call.from_user.id):
        return await call.answer("⏳ در حال انجام عملیات قبلی...", show_alert=True)
    await set_busy(call.from_user.id, "buy")
    try:
        # prevent multiple active orders
        from repositories.orders_repo import get_active_order
        active = await get_active_order(call.from_user.id)
        if active:
            await call.answer(f"⏳ شما یک سفارش فعال دارید (#{active['id']}).", show_alert=True)
            return

        flower_id = int(call.data.split(":")[1])
        f = await get_flower(flower_id)
        if not f or int(f.get("stock") or 0) <= 0 or int(f.get("is_active") or 0) != 1:
            await call.answer("این آگهی موجود نیست.", show_alert=True)
            return

        max_qty = min(int(f["stock"]), 5)
        await state.update_data(flower_id=flower_id)
        await state.set_state(BuyFlow.qty)
        await call.message.answer("تعداد را انتخاب کنید:", reply_markup=choose_qty_kb(max_qty))
        await call.answer()
    finally:
        await clear_busy(call.from_user.id)

@router.callback_query(F.data.startswith("qty:"))
async def buy_qty(call: CallbackQuery, state: FSMContext):
    if await is_busy(call.from_user.id):
        return await call.answer("⏳ در حال پردازش...", show_alert=True)
    await set_busy(call.from_user.id, "qty")
    try:
        qty = int(call.data.split(":")[1])
        data = await state.get_data()
        flower_id = data.get("flower_id")
        if not flower_id:
            await call.answer("خطا. دوباره شروع کنید.", show_alert=True); return
        f = await get_flower(flower_id)
        if not f or int(f["stock"]) < qty:
            await call.answer("موجودی کافی نیست.", show_alert=True); return

        total = int(f["price_toman"]) * qty
        # decrement stock immediately (reserved)
        await dec_stock(flower_id, qty)

        order_id = await create_order(call.from_user.id, f["seller_id"], flower_id, qty, total)
        await state.update_data(order_id=order_id)
        await state.set_state(BuyFlow.receipt)

        await call.message.answer(
            "✅ سفارش ثبت شد.\n"
            f"🧾 شماره سفارش: <b>{order_id}</b>\n"
            f"💰 مبلغ: <b>{money(total)}</b> تومان\n\n"
            "📌 لطفاً مبلغ را کارت به کارت کنید و سپس رسید را ارسال کنید.\n"
            f"💳 کارت فروشنده: <code>{f.get('seller_card') or '-'}</code>\n"
            f"👤 به نام: <b>{f.get('seller_owner') or '-'}</b>\n\n"
            "📷 حالا عکس رسید را ارسال کنید:"
        )
        await call.answer()
    finally:
        await clear_busy(call.from_user.id)

@router.message(BuyFlow.receipt, F.photo)
async def buy_receipt(message: Message, state: FSMContext):
    if await is_busy(message.from_user.id):
        return await message.answer("⏳ در حال پردازش...")
    await set_busy(message.from_user.id, "receipt")
    try:
        data = await state.get_data()
        order_id = data.get("order_id")
        if not order_id:
            await state.clear(); return await message.answer("خطا. دوباره شروع کنید.")
        receipt_id = message.photo[-1].file_id
        await set_receipt(order_id, receipt_id)

        await message.answer("✅ رسید ثبت شد. منتظر تایید فروشنده باشید.", reply_markup=buyer_home_kb())

        # notify seller with actions
        order = await get_order(order_id)
        cap = (
            "🧾 <b>سفارش جدید</b>\n"
            f"Order: <b>{order_id}</b>\n"
            f"Buyer: <code>{order['buyer_id']}</code>\n"
            f"Qty: <b>{order['qty']}</b>\n"
            f"Total: <b>{money(int(order['total_price']))}</b> تومان\n"
            "📌 رسید ارسال شده است. تایید/رد کنید."
        )
        await message.bot.send_photo(order["seller_id"], receipt_id, caption=cap, reply_markup=receipt_actions_kb(order_id))
        await state.clear()
    finally:
        await clear_busy(message.from_user.id)

@router.message(BuyFlow.receipt)
async def buy_receipt_wrong(message: Message):
    await message.answer("⚠️ لطفاً فقط <b>عکس رسید</b> را ارسال کنید.")

@router.message(F.location)
async def buyer_location(message: Message, state: FSMContext):
    # accept location only if there is an accepted order
    from repositories.orders_repo import get_active_order
    order = await get_active_order(message.from_user.id)
    if not order or order["status"] != "accepted":
        return

    await set_location(order["id"], message.location.latitude, message.location.longitude)
    await state.set_state(BuyFlow.address)
    await state.update_data(order_id=order["id"])
    await message.answer("✅ لوکیشن ثبت شد. حالا آدرس دقیق (متنی) را ارسال کنید:")

@router.message(BuyFlow.address)
async def buyer_address(message: Message, state: FSMContext):
    data = await state.get_data()
    order_id = data.get("order_id")
    if not order_id:
        await state.clear()
        return await message.answer("خطا. دوباره شروع کنید.")
    addr = (message.text or "").strip()
    await set_address(order_id, addr)

    await message.answer("✅ آدرس ثبت شد. فروشنده در حال ارسال است.", reply_markup=buyer_home_kb())
    order = await get_order(order_id)
    await message.bot.send_message(order["seller_id"], f"📦 سفارش #{order_id} آماده ارسال است.\nآدرس: {addr}")
    await state.clear()

@router.callback_query(F.data.startswith("buyer_confirm:"))
async def buyer_confirm_delivery(call: CallbackQuery):
    order_id = int(call.data.split(":")[1])
    order = await get_order(order_id)
    if not order or order["buyer_id"] != call.from_user.id:
        return await call.answer("دسترسی ندارید", show_alert=True)
    from repositories.orders_repo import mark_completed
    await mark_completed(order_id)
    await call.message.edit_text("✅ تحویل تایید شد. سفارش تکمیل شد.")
    await call.answer()

@router.callback_query(F.data.startswith("buyer_issue:"))
async def buyer_issue_delivery(call: CallbackQuery):
    order_id = int(call.data.split(":")[1])
    order = await get_order(order_id)
    if not order or order["buyer_id"] != call.from_user.id:
        return await call.answer("دسترسی ندارید", show_alert=True)
    await call.message.edit_text("⚠️ مشکل ثبت شد. پشتیبانی با شما تماس می‌گیرد.")
    await call.answer()


@router.callback_query(F.data=='qty_cancel')
async def qty_cancel(call: CallbackQuery, state: FSMContext):
    await state.clear()
    await call.message.answer('✅ لغو شد.', reply_markup=buyer_home_kb())
    await call.answer()