
from aiogram import Router, F
from aiogram.types import Message, CallbackQuery, InlineKeyboardMarkup, InlineKeyboardButton

from utils.filters import ModeFilter
from config import settings
from db import fetchall, fetchone, execute
from keyboards.admin import admin_home_kb, admin_sellers_kb

router = Router()
router.message.filter(ModeFilter('admin'))
router.callback_query.filter(ModeFilter('admin'))

def is_admin(uid:int)->bool:
    return uid in settings.ADMIN_IDS

async def deny(message: Message):
    return await message.answer("⛔ دسترسی ندارید.", reply_markup=admin_home_kb())

def seller_row_kb(user_id:int):
    return InlineKeyboardMarkup(inline_keyboard=[
        [
            InlineKeyboardButton(text="✅ تایید", callback_data=f"admin_seller_ok:{user_id}"),
            InlineKeyboardButton(text="❌ رد", callback_data=f"admin_seller_no:{user_id}"),
        ]
    ])

@router.message(F.text == "✅ تایید فروشنده‌ها")
async def admin_sellers_entry(message: Message):
    if not is_admin(message.from_user.id):
        return await deny(message)
    await message.answer("پنل فروشنده‌ها:", reply_markup=admin_sellers_kb())

@router.message(F.text == "✅ لیست در انتظار تایید")
async def admin_pending_sellers(message: Message):
    if not is_admin(message.from_user.id):
        return await deny(message)
    rows = await fetchall("SELECT user_id, shop_name, city, phone, created_at FROM sellers WHERE status='pending' ORDER BY created_at ASC LIMIT 50")
    if not rows:
        return await message.answer("✅ فروشنده‌ی در انتظار نداریم.", reply_markup=admin_sellers_kb())
    for r in rows:
        text = (
            "⏳ <b>فروشنده در انتظار</b>\n\n"
            f"👤 <code>{r['user_id']}</code>\n"
            f"🏪 {r.get('shop_name') or '-'}\n"
            f"🏙 {r.get('city') or '-'}\n"
            f"📱 {r.get('phone') or '-'}\n"
        )
        await message.answer(text, reply_markup=seller_row_kb(r['user_id']))
    await message.answer("پایان لیست ✅", reply_markup=admin_sellers_kb())

@router.message(F.text == "📦 لیست فروشنده‌های فعال")
async def admin_active_sellers(message: Message):
    if not is_admin(message.from_user.id):
        return await deny(message)
    rows = await fetchall("SELECT user_id, shop_name, city, phone, updated_at FROM sellers WHERE status='active' ORDER BY updated_at DESC LIMIT 50")
    if not rows:
        return await message.answer("فروشنده فعال نداریم.", reply_markup=admin_sellers_kb())
    text = "🧑‍🌾 <b>فروشنده‌های فعال</b>\n\n"
    for r in rows:
        text += f"👤 <code>{r['user_id']}</code> | {r.get('shop_name') or '-'} | {r.get('city') or '-'} | {r.get('phone') or '-'}\n"
    await message.answer(text, reply_markup=admin_sellers_kb())

@router.callback_query(F.data.startswith("admin_seller_ok:"))
async def admin_seller_ok(call: CallbackQuery):
    if not is_admin(call.from_user.id):
        return await call.answer("⛔", show_alert=True)
    uid = int(call.data.split(":")[1])
    s = await fetchone("SELECT status FROM sellers WHERE user_id=%s", (uid,))
    if not s:
        await call.answer("یافت نشد", show_alert=True); return
    await execute("UPDATE sellers SET status='active', updated_at=NOW() WHERE user_id=%s", (uid,))
    await call.answer("✅ تایید شد")
    try:
        await call.bot.send_message(uid, "✅ درخواست فروشندگی شما تایید شد.\nبرای ورود به پنل فروشنده دستور /seller را بزنید.")
    except Exception:
        pass
    await call.message.edit_text("✅ تایید شد")

@router.callback_query(F.data.startswith("admin_seller_no:"))
async def admin_seller_no(call: CallbackQuery):
    if not is_admin(call.from_user.id):
        return await call.answer("⛔", show_alert=True)
    uid = int(call.data.split(":")[1])
    s = await fetchone("SELECT status FROM sellers WHERE user_id=%s", (uid,))
    if not s:
        await call.answer("یافت نشد", show_alert=True); return
    await execute("UPDATE sellers SET status='rejected', updated_at=NOW() WHERE user_id=%s", (uid,))
    await call.answer("❌ رد شد")
    try:
        await call.bot.send_message(uid, "❌ درخواست فروشندگی شما رد شد.\nدر صورت نیاز دوباره /seller_apply ارسال کنید.")
    except Exception:
        pass
    await call.message.edit_text("❌ رد شد")

@router.message(F.text == "⬅️ بازگشت به منوی اصلی")
async def admin_back(message: Message):
    if not is_admin(message.from_user.id):
        return await deny(message)
    await message.answer("🅰️ منوی ادمین", reply_markup=admin_home_kb())
