import aiomysql
from contextlib import asynccontextmanager
from config import settings

pool: aiomysql.Pool | None = None

async def init_db():
    global pool
    pool = await aiomysql.create_pool(
        host=settings.DB_HOST,
        port=settings.DB_PORT,
        user=settings.DB_USER,
        password=settings.DB_PASS,
        db=settings.DB_NAME,
        minsize=1,
        maxsize=10,
        autocommit=True,
        charset="utf8mb4",
    )

async def fetchone(q, p=None):
    async with pool.acquire() as conn:
        async with conn.cursor(aiomysql.DictCursor) as cur:
            await cur.execute(q, p or ())
            return await cur.fetchone()

async def fetchall(q, p=None):
    async with pool.acquire() as conn:
        async with conn.cursor(aiomysql.DictCursor) as cur:
            await cur.execute(q, p or ())
            return await cur.fetchall()

async def execute(q, p=None):
    async with pool.acquire() as conn:
        async with conn.cursor() as cur:
            await cur.execute(q, p or ())
            return cur.lastrowid

@asynccontextmanager
async def transaction():
    async with pool.acquire() as conn:
        await conn.begin()
        try:
            async with conn.cursor(aiomysql.DictCursor) as cur:
                yield conn, cur
            await conn.commit()
        except Exception:
            await conn.rollback()
            raise
